<?php

use Illuminate\Database\Migrations\Migration;
use Spatie\Permission\Models\Permission;
use Spatie\Permission\Models\Role;
use Spatie\Permission\PermissionRegistrar;

return new class extends Migration
{
    /**
     * Assign contact view permissions to specific roles.
     */
    public function up(): void
    {
        if (! class_exists(Role::class) || ! class_exists(Permission::class)) {
            return;
        }

        $this->forgetPermissionCache();

        foreach ($this->getRolePermissions() as $roleName => $permissionNames) {
            $role = $this->getRole($roleName);
            if (! $role) {
                continue;
            }

            foreach ($permissionNames as $permissionName) {
                $permission = $this->getPermission($permissionName);
                if (! $permission) {
                    continue;
                }

                $role->givePermissionTo($permission);
            }
        }

        $this->forgetPermissionCache();
    }

    /**
     * Revoke the contact view permissions from the specific roles.
     */
    public function down(): void
    {
        if (! class_exists(Role::class) || ! class_exists(Permission::class)) {
            return;
        }

        $this->forgetPermissionCache();

        foreach ($this->getRolePermissions() as $roleName => $permissionNames) {
            $role = $this->getRole($roleName);
            if (! $role) {
                continue;
            }

            foreach ($permissionNames as $permissionName) {
                $permission = $this->getPermission($permissionName);
                if (! $permission) {
                    continue;
                }

                $role->revokePermissionTo($permission);
            }
        }

        $this->forgetPermissionCache();
    }

    private function getRolePermissions(): array
    {
        return [
            'CSR - BPDI Sta Cruz' => ['customer.view'],
            'Purchasing Assistant' => ['customer.view', 'supplier.view'],
            'pharmacist' => ['customer.view', 'supplier.view'],
        ];
    }

    private function getRole(string $name): ?Role
    {
        return Role::where('name', $name)
            ->where('guard_name', 'web')
            ->first();
    }

    private function getPermission(string $name): ?Permission
    {
        return Permission::where('name', $name)
            ->where('guard_name', 'web')
            ->first();
    }

    private function forgetPermissionCache(): void
    {
        if (class_exists(PermissionRegistrar::class)) {
            app(PermissionRegistrar::class)->forgetCachedPermissions();
        }
    }
};
